/**
* \file: errmem_interface.h
*
* This file keeps definitions to define general funcitonality
* needed by the error memory daemon.
* 
* \component: errmemd
*
* \author: Kai Tomerius (ktomerius@de.adit-jv.com)
*          Markus Kretschmann (mkretschmann@de.adit-jv.com)
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
* <history item>
*/

#ifndef _ERRMEM_INTERFACE_H
#define _ERRMEM_INTERFACE_H

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <linux/errmem.h>

#define ERRMEM_REQUIRED_VERSION_MIN 0x100 // v1.00
#define ERRMEM_REQUIRED_VERSION_MAX 0x399 // v3.99

/* Forward declaration */
struct Errmemd;
struct ErrmemConnect;

typedef struct ErrmemInterface {
	char* name;
	int efd;
	void *con;
	unsigned maxsize;
	unsigned low;
	int flush;
	unsigned nr_drv_slots;
	unsigned char* buffer;
} ErrmemInterface_t;

// work on notifications from error memory frontend
int32_t errmem_handle_frontend(struct Errmemd *d, struct ErrmemConnect* c);

// connect to the error memory frontend to be notified if messages are available
int32_t errmem_connect_frontend(struct Errmemd* d);

// disconnect from the error memroy frontend to not be notified by events
int32_t errmem_disconnect_frontend(struct Errmemd* d);

// errmem_create - initialize the error memory interface
ErrmemInterface_t* errmem_create(char* driver_dev);

// errmem_info - store information about the error memory interface
void errmem_info(ErrmemInterface_t* b, FILE* fd);

// errmem_get_kernel_message_level - get a driver parameter
unsigned errmem_get_kernel_message_level(ErrmemInterface_t* b);

// errmem_get_watermark_low - get a driver parameter
unsigned errmem_get_watermark_low(ErrmemInterface_t* b);

// errmem_get_watermark_high - get a driver parameter
unsigned errmem_get_watermark_high(ErrmemInterface_t* b);

// errmem_get_number_of_slots - get a driver parameter
unsigned errmem_get_number_of_slots(ErrmemInterface_t* b);

// errmem_get_slotsize - get a driver parameter
unsigned errmem_get_slotsize(ErrmemInterface_t* b);

// errmem_get_used_slots - get a driver parameter
unsigned errmem_get_used_slots(ErrmemInterface_t* b);

// errmem_get_version - get a driver parameter
unsigned long errmem_get_version(ErrmemInterface_t* b);

// errmem_set_kernel_message_level - set a driver parameter
void errmem_set_kernel_message_level(ErrmemInterface_t* b, unsigned);

// errmem_set_watermark_low - set a driver parameter
int32_t errmem_set_watermark_low(ErrmemInterface_t* b, unsigned);

// errmem_set_watermark_high - set a driver parameter
int32_t errmem_set_watermark_high(ErrmemInterface_t* b, unsigned);

// errmem_destroy - de-initialize the error memory interface
void errmem_destroy(ErrmemInterface_t* b);

//----------------------------------------------------------------------
// helper routines for debugging

// errmem_fill - put stuff into the error memory device
void errmem_fill(const char* errmem, const char* source);

#endif // ifndef _ERRMEM_INTERFACE_H
